#include "config.h"
#include "LinQ.h"
#include "LinqMain.h"
#include "LinqUsers.h"
#include "SendFileDlg.h"
#include "icqdb.h"
#include "intl.h"
#include <stdio.h>
#include <sys/stat.h>

#ifdef G_OS_WIN32
#include <dir.h>  // for mkdir()
#endif

const char *status2text(uint32 status)
{
	switch (status) {
	case STATUS_ONLINE: return _("Online");
	case STATUS_AWAY: return _("Away");
	case STATUS_INVIS: return _("Invisible");
	case STATUS_OFFLINE: return _("Offline");
	}

	return _("Unknown");
}

LinqMain *linqMain;


LinqUser::LinqUser()
{
	login_status = STATUS_ONLINE;
	now_status = STATUS_OFFLINE;
}

LinqMain::LinqMain(SocketRegistry *reg):ICQMain(reg)
{
	linqMain = this;
	main_win = NULL;
	send_keepalive_timeout = 0;
	check_send_queue_timeout = 0;
}

LinqMain::~LinqMain()
{
	linqMain = NULL;
}

void LinqMain::getUserDir(std::string &userdir, const gchar *user)
{
#ifdef G_OS_WIN32
	userdir = linq_data_dir;
	userdir += G_DIR_SEPARATOR_S"usrs"G_DIR_SEPARATOR_S;
	userdir += user;
	userdir += G_DIR_SEPARATOR;
#else
	userdir = g_get_home_dir();
	userdir += "/.linq"G_DIR_SEPARATOR_S;
#endif	
}

void LinqMain::doLogin()
{
	skin = new Skin;
	
	// initialize the user dir.
	struct stat st;
	string pathName;
#ifdef G_OS_WIN32
	pathName = linq_data_dir;
	pathName += G_DIR_SEPARATOR_S"usrs";
	if (stat (pathName.c_str(), &st)) {
		mkdir (pathName.c_str());
	}
#else
	pathName = g_get_home_dir();
	pathName += "/.linq";
	if (stat (pathName.c_str(), &st))
		mkdir (pathName.c_str(),00755);
#endif
		
	LinqUsers::setDir(pathName.c_str());
	pathName += G_DIR_SEPARATOR_S"usrs.idx";
	if (!g_file_test(pathName.c_str(),G_FILE_TEST_EXISTS))
		ShowRegWizard();
	else
		ShowLoginDlg();
	gtk_main();
}

void LinqMain::Quit()
{	
	destroyUser();
	
	if (main_win)
		delete main_win;
	delete skin;	
	gtk_main_quit();
}

void LinqMain::ShowLoginDlg()
{
	LinqWindow *win;
	win = findWindow(WIN_LOGIN);
	LoginDlg *dlg;
	if (win) {
		dlg = (LoginDlg *)win;
	}
	else {
		dlg = new LoginDlg;
	}
	dlg->Show();		
}

void LinqMain::ShowSearchDlg()
{
	LinqWindow *win;
	win = findWindow(WIN_SEARCH);
	SearchDlg *dlg;
	if (win) {
		dlg = (SearchDlg *)win;
	}
	else {
		dlg = new SearchDlg;
	}
	dlg->Show();		
}

void LinqMain::ShowRegWizard(const gchar *user, const gchar *passwd)
{
	LinqWindow *win;
	win = findWindow(WIN_REG_WIZARD);
	RegWizard *dlg;
	if (win) {
		dlg = (RegWizard *)win;
	}
	else {
		dlg = new RegWizard;
	}
	dlg->Show(user, passwd);		
}

void LinqMain::ShowMainWin()
{
	if (!main_win)
		main_win = new MainWin;
	main_win->Show();
}

LinqWindow *LinqMain::findWindow(int type)
{
	std::list<LinqWindow *>::iterator iter;
	for (iter = windowList.begin(); iter != windowList.end(); ++iter) {
		LinqWindow *win = (LinqWindow *) *iter;
		if (win->type == type)
			return win;
	}
	return NULL;
}

void LinqMain::destroyUser()
{
	if (myInfo.now_status == STATUS_ONLINE)
		logout();
	while (!windowList.empty())
		delete (LinqWindow *) windowList.front(); //it will remove from windowList when be deleted.
	
	onlineInfoList.clear();
}

void LinqMain::logLine(const char *fmt, ...)
{
/*	char buf[1024];

	va_list args;
	va_start(args, fmt);

	vsnprintf(buf, sizeof(buf), fmt, args);

	va_end(args);
	g_print("%s",buf);*/
}

const char *LinqMain::getUserName()
{
	return myInfo.name.c_str();
}

bool LinqMain::getContactInfo(ONLINE_INFO &info)
{
	LinqOnlineInfo *c = findOnlineInfo(info.name);
	if (!c)
		return false;

	info.status = c->status;
	info.tcp_ver = c->tcp_ver;
	info.ip = c->ip;
	info.real_ip = c->real_ip;
	info.msg_port = c->msg_port;
	return true;
}

TCPSessionListener *LinqMain::getSessionListener(const char *type, TCPSessionBase *session)
{
	TCPSessionListener *l = ICQMain::getSessionListener(type, session);
	if (!l) {
		if (strcmp(type, "SENDFILE") == 0)
			l = new CSendFileDlg(session);
	}
	return l;
}

LinqOnlineInfo *LinqMain::findOnlineInfo(const char *name)
{
	std::list<LinqOnlineInfo>::iterator iter;
	for (iter = onlineInfoList.begin(); iter!= onlineInfoList.end(); ++iter) {
		LinqOnlineInfo *c = &(*iter);
		if (c->name == name)
			return c;
	}
	return NULL;
}

gboolean LinqMain::send_keepalive_cb(gpointer data)
{
	linqMain->getUDPSession()->sendKeepAlive();
	return true;
}

gboolean LinqMain::check_send_queue_cb(gpointer data)
{
	linqMain->getUDPSession()->checkSendQueue();
	return true;
}

void LinqMain::onAck(uint32 seq)
{
	logLine("packet %u is acked", seq);
}

void LinqMain::onSendError(uint16 seq)
{
	logLine("packet %d failed", seq);
}

void LinqMain::onConnect(bool connected)
{
	if (connected) {
		check_send_queue_timeout = gtk_timeout_add(SEND_TIMEOUT * 1000, check_send_queue_cb, this);		
	}
	else {
		if (check_send_queue_timeout) {
			gtk_timeout_remove(check_send_queue_timeout);
			check_send_queue_timeout = 0;
		}
		if (send_keepalive_timeout) {
			gtk_timeout_remove(send_keepalive_timeout);
			send_keepalive_timeout = 0;
		}
	}		
	
	LinqWindow *win;
	win = linqMain->findWindow(WIN_REG_WIZARD);	
	if (win) {
		RegWizard *dlg;
		dlg = (RegWizard *)win;
		dlg->onConnect(connected);
	}
	if (main_win)
		main_win->onConnect(connected);
}

void LinqMain::onRegisterReply(uint8 error)
{
	LinqWindow *win;
	win = linqMain->findWindow(WIN_REG_WIZARD);	
	if (win) {
		RegWizard *dlg;
		dlg = (RegWizard *)win;
		dlg->onRegisterReply(error);
	}
}

void LinqMain::onPreLoginReply(uint16 sequence, const char *token)
{
	LinqWindow *win;
	win = linqMain->findWindow(WIN_REG_WIZARD);	
	if (win) {
		RegWizard *dlg;
		dlg = (RegWizard *)win;
		if (dlg->onPreLoginReply(sequence, token))
			return;
	}

	if (main_win)
		main_win->onPreLoginReply(sequence, token);
}

void LinqMain::onLoginReply(uint8 error)
{
	if (error == LOGIN_SUCCESS) {
		send_keepalive_timeout = gtk_timeout_add(KEEPALIVE_TIMEOUT * 1000, send_keepalive_cb, this);
	}
	
	LinqWindow *win;
	win = linqMain->findWindow(WIN_REG_WIZARD);	
	if (win) {
		RegWizard *dlg;
		dlg = (RegWizard *)win;
		dlg->onLoginReply(error);
	}
	if (main_win)
		main_win->onLoginReply(error);
}

void LinqMain::onAddContactReply(const char *name, uint8 auth)
{
	LinqWindow *win;
	win = linqMain->findWindow(WIN_SEARCH);
	gboolean processed = false;
	if (win) {
		SearchDlg *dlg;
		dlg = (SearchDlg *)win;
		processed = dlg->onAddContactReply(name, auth);
	}
	if (!processed) {
		if (main_win)
			main_win->onAddContactReply(name, auth);
	}
}

void LinqMain::onContactListReply(const char *result[], int n)
{
	LinqWindow *win;
	win = linqMain->findWindow(WIN_REG_WIZARD);
	if (win) {
		RegWizard *dlg;
		dlg = (RegWizard *)win;
		dlg->onContactListReply(result, n);
	}
	if (main_win)
		main_win->onContactListReply(result, n);	
}

void LinqMain::onUserOnline(ONLINE_INFO &info)
{
	LinqOnlineInfo *c;
	std::list<LinqOnlineInfo>::iterator iter;
	for (iter = onlineInfoList.begin(); iter!= onlineInfoList.end(); ++iter) {
		c = &(*iter);
		if (c->name == info.name) {
			onlineInfoList.erase(iter);
			break;
		}
	}

	onlineInfoList.push_front(info);

	if (main_win)
		main_win->onUserOnline(info.name);
/*	in_addr addr;
	addr.s_addr = htonl(info.ip);
	string ip = inet_ntoa(addr);

	addr.s_addr = htonl(info.real_ip);
	string real_ip = inet_ntoa(addr);

	logLine("%s goes online @ %s(%s):%d", info.name, (LPCTSTR) ip, (LPCTSTR) real_ip, info.msg_port);*/
}

void LinqMain::onUserOffline(const char *name)
{
	std::list<LinqOnlineInfo>::iterator iter;
	for (iter = onlineInfoList.begin(); iter!= onlineInfoList.end(); ++iter) {
		LinqOnlineInfo *c = &(*iter);
		if (c->name == name) {
			onlineInfoList.erase(iter);
			break;
		}
	}	
	logLine("%s goes offline", name);
	if (main_win)
		main_win->onUserOffline(name);
}

void LinqMain::onUserStatus(const char *name, uint32 status)
{
	logLine("%s changed status to %s.\n", name, status2text(status));
	if (main_win)
		main_win->onUserStatus(name, status);
}

void LinqMain::onSearchResult(SEARCH_RESULT result[], int n)
{
	LinqWindow *win;
	win = linqMain->findWindow(WIN_SEARCH);
	if (win) {
		SearchDlg *dlg;
		dlg = (SearchDlg *)win;
		dlg->onSearchResult(result, n);
	}
}

void LinqMain::onRecvMessage(uint8 type, const char *from, time_t when, const char *text)
{
	ICQMain::onRecvMessage(type, from, when, text);
	if (main_win)
		main_win->onRecvMessage(type, from, when, text);
}

void LinqMain::onContactInfoReply(CONTACT_INFO &c)
{
	LinqContactInfo contact(c);
	IcqDB::saveContactInfo(contact);
	
	if (main_win)
		main_win->onContactInfoReply(c);
}

void LinqMain::onUserInfoReply(USER_INFO &user)
{
	LinqUserInfo contact(user);
	IcqDB::saveUserInfo(contact);
	
	LinqWindow *win;
	win = linqMain->findWindow(WIN_REG_WIZARD);
	if (win) {
		RegWizard *dlg;
		dlg = (RegWizard *)win;
		dlg->onUserInfoReply(user);
	}

	if (main_win)
		main_win->onUserInfoReply(user);
}

void LinqMain::onPreChangePasswordReply(uint16 sequence, const char *token)
{
	std::list<LinqWindow *>::iterator iter;
	for (iter = windowList.begin(); iter != windowList.end(); ++iter) {
		LinqWindow *win;
		win = (LinqWindow *) *iter;
		if (win->type == WIN_VIEW_DETAIL) {
			UserInfoDlg *dlg = (UserInfoDlg *) *iter;
			if (dlg->onPreChangePasswordReply(sequence, token))
				return;
		}
	}
}
