/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   copyright            : (C) 2003 by Zhang Yong                         *
 *   email                : z-yong163@163.com                              *
 ***************************************************************************/

#include "config.h"
#include "icqmain.h"
#include "tcpsession.h"
#include "messagesession.h"
#include "textstream.h"
#include "nullproxy.h"
#include "socksudpproxy.h"
#include "sockstcpproxy.h"
#include "httpproxy.h"
#include "icqlog.h"
#include <assert.h>


#define MESSAGE_SESSION		"MESSAGE"


SocketRegistry *socketRegistry;


ICQMain::ICQMain(SocketRegistry *reg)
{
	socketRegistry = reg;

	proxyInfo = NULL;
	udpSession = new UDPSession(this);

#ifndef DEBUG
	srand(time(NULL));
#endif
}

ICQMain::~ICQMain()
{
	if (proxyInfo)
		delete proxyInfo;

	destroyAllSessions();
	delete udpSession;
}

TCPSessionListener *ICQMain::getSessionListener(const char *type, TCPSessionBase *session)
{
	if (strcmp(type, MESSAGE_SESSION) == 0)
		return new MessageSession((TCPSession *) session);
	return NULL;
}

void ICQMain::setProxy(PROXY_INFO *proxy)
{
	if (proxyInfo) {
		delete proxyInfo;
		proxyInfo = NULL;
	}

	if (proxy) {
		proxyInfo = new PROXY_INFO;
		*proxyInfo = *proxy;
	}
}

void ICQMain::connect(const char *host, uint16 port)
{
	Socket *sock;

	if (!proxyInfo)
		sock = new NullProxy;
	else if (proxyInfo->type == PROXY_SOCKS5)
		sock = new SocksUDPProxy(*proxyInfo);
	else
		sock = new HTTPProxy(*proxyInfo, udpSession);

	udpSession->connect(sock, host, port);
}

TCPSession *ICQMain::findSession(const char *type, const char *name)
{
	list<TCPSession *>::iterator it;

	for (it = tcpSessionList.begin(); it != tcpSessionList.end(); ++it) {
		TCPSession *s = *it;
		if (s->sessionType == type && s->contactName == name)
			return s;
	}
	return NULL;
}

uint16 ICQMain::createListenSession(const char *type)
{
	TCPSession *session = NULL;

	list<TCPSession *>::iterator it;
	for (it = tcpSessionList.begin(); it != tcpSessionList.end(); ++it) {
		TCPSession *s = *it;
		if (s->isListen() && s->sessionType == type) {
			session = s;
			break;
		}
	}

	int fd;
	sockaddr_in addr;

	if (!session) {
		fd = socket(AF_INET, SOCK_STREAM, 0);
		assert(fd >= 0);

		memset(&addr, 0, sizeof(addr));
		addr.sin_family = AF_INET;
		addr.sin_addr.s_addr = INADDR_ANY;
		addr.sin_port = 0;
		bind(fd, (sockaddr *) &addr, sizeof(addr));

		Socket *sock = new NullProxy(fd);
		session = new TCPSession(this, sock, type, "", false);

		session->listen();
		tcpSessionList.push_back(session);
	}

	fd = session->getSocketFd();

	socklen_t len = sizeof(addr);
	getsockname(fd, (sockaddr *) &addr, &len);

	return ntohs(addr.sin_port);
}

TCPSession *ICQMain::createTCPSession(const char *type, const char *to, uint16 port, bool isSend)
{
	ONLINE_INFO c;
	c.name = to;
	if (!getContactInfo(c))
		return NULL;

	Socket *sock;
	PROXY_INFO info;

	if (proxyInfo && proxyInfo->type == PROXY_SOCKS5)
		sock = new SocksTCPProxy(info);
	else
		sock = new NullProxy;

	TCPSession *session = new TCPSession(this, sock, type, to, isSend);
	session->connect(c.real_ip, port);

	tcpSessionList.push_back(session);
	return session;
}

TCPSession *ICQMain::acceptSession(const char *type, Socket *sock)
{
	int fd = accept(sock->getFd(), NULL, NULL);
	if (fd < 0)
		return NULL;

	sock = new NullProxy(fd);
	TCPSession *session = new TCPSession(this, sock, type, "", false);
	tcpSessionList.push_back(session);

	return session;
}

void ICQMain::removeSession(TCPSession *s)
{
	tcpSessionList.remove(s);
}

void ICQMain::destroyAllSessions()
{
	list<TCPSession *>::iterator it;
	for (it = tcpSessionList.begin(); it != tcpSessionList.end(); ++it)
		delete *it;

	tcpSessionList.clear();
}

void ICQMain::login(const char *name, const char *passwd, const char *token, uint16 sequence, uint32 status)
{
	uint16 port = createListenSession(MESSAGE_SESSION);

	udpSession->login(name, passwd, token, sequence, status, port);
}

void ICQMain::logout()
{
	udpSession->logout();

	destroyAllSessions();
}

uint32 ICQMain::sendMessage(const char *to, const char *text)
{
	ONLINE_INFO c;
	c.name = to;
	if (!getContactInfo(c))
		return 0;

	if (!udpSession->isBehindWall() && (c.ip != c.real_ip))
		return 0;

	TCPSession *session = findSession(MESSAGE_SESSION, to);
	if (!session) {
		session = createTCPSession(MESSAGE_SESSION, to, c.msg_port, true);
		session->setListener(new MessageSession(session));
	}

	if (!session->isConnectable())
		return 0;

	MessageSession *s = (MessageSession *) session->listener;
	return s->sendMessage(text);
}

void ICQMain::sendTCPRequest(const char *type, const char *to, const char *nick, const char *reason)
{
	ONLINE_INFO c;
	c.name = to;
	if (!getContactInfo(c))
		return;

	uint16 port = 0;

	if (!udpSession->isBehindWall() && c.ip != c.real_ip)
		port = createListenSession(type);

	TextOutStream text;
	text << type << port << nick << reason;
	udpSession->sendMessage(MSG_TCP_REQUEST, to, text);
}

void ICQMain::acceptTCPRequest(const char *type, const char *to, uint16 port)
{
	if (port)
		createTCPSession(type, to, port, false);
	else {
		port = createListenSession(type);

		TextOutStream out;
		out << type << port;
		udpSession->sendMessage(MSG_TCP_ACCEPT, to, out);
	}
}

void ICQMain::onRecvMessage(uint8 type, const char *from, time_t when, const char *text)
{
	if (type == MSG_TCP_ACCEPT) {
		TextInStream in(text);

		string name;
		uint16 port;
		in >> name >> port;

		if (port)
			createTCPSession(name.c_str(), from, port, true);
	}
}
